const LICENSE_URL = "https://kevin-dennis.de/licenses.txt";

async function validateLicenseInBackground() {
    const { licenseKey } = await chrome.storage.local.get('licenseKey');
    if (!licenseKey) {
        await chrome.storage.local.set({ isLicensed: false });
        return false;
    }

    try {
        const response = await fetch(LICENSE_URL, { cache: "no-store" });
        if (!response.ok) {
            console.error("Lizenz-Server nicht erreichbar. Nächster Versuch beim nächsten Intervall.");
            // Wir nehmen an, die Lizenz ist vorerst noch gültig, um Offline-Nutzung zu ermöglichen.
            return true; 
        }

        const text = await response.text();
        const keyDataLine = text.split(/\r?\n/).find(line => line.split('|')[0].trim() === licenseKey);

        if (!keyDataLine) throw new Error('Lizenzschlüssel serverseitig entfernt.');

        const [, licenseExpiry] = keyDataLine.split('|').map(s => s.trim());
        const [day, month, year] = licenseExpiry.split('.');
        const expiryDate = new Date(`${year}-${month}-${day}`);
        const today = new Date();
        today.setHours(0, 0, 0, 0);

        if (expiryDate < today) throw new Error('Lizenz ist abgelaufen.');

        await chrome.storage.local.set({ isLicensed: true });
        return true;

    } catch (error) {
        console.error(`Lizenzprüfung im Hintergrund fehlgeschlagen: ${error.message}`);
        await chrome.storage.local.remove(['isLicensed', 'licenseKey', 'licenseUser', 'licenseExpiry']);
        return false;
    }
}

// Stabile Funktion zum Abrufen der Daten über einen Tab
function fetchDataViaTab() {
    const EXPRESS_URL = "https://me-hpgermany.tickets.heide-park.de/fastlanewizardstatic/ExpressTicketDE";
    const API_URL = "https://me-hpgermany.tickets.heide-park.de/api/request/getkeywordcalendar";
    const API_PAYLOAD = { "display_zero_capacity": "0", "keyword": "ExpressTicketDE", "end_date": "2025-11-02", "start_date": "2025-10-13", "request_type": "GetKeywordCalendar", "_version": "6.17.9", "application_id": "1500", "merchant_id": "6500", "machine_id": "500", "agent_id": "5", "user_id": "5", "device": "desktop", "language": "de-de", "request_token": "5B98CF43851DAD16D9C16799EE4998EB", "cart_id": "536057884", "cart_key": "1022794776", "session_id": "runi.070744.778" };

    return new Promise((resolve, reject) => {
        chrome.tabs.create({ url: EXPRESS_URL, active: false }, (tab) => {
            if (!tab || !tab.id) {
                return reject(new Error("Tab konnte nicht erstellt werden."));
            }
            const listener = (tabId, changeInfo) => {
                if (tabId === tab.id && changeInfo.status === 'complete') {
                    chrome.tabs.onUpdated.removeListener(listener);
                    chrome.scripting.executeScript({
                        target: { tabId },
                        // KORRIGIERT: Robuste Fehlerbehandlung innerhalb des Scripts hinzugefügt
                        func: (apiUrl, apiPayload) => fetch(apiUrl, {
                            method: "POST",
                            headers: { "accept": "application/json, text/plain, */*", "content-type": "application/json;charset=UTF-8", "com-accessopassport-app-id": "1500", "com-accessopassport-client": "accesso26", "com-accessopassport-language": "de-de", "com-accessopassport-merchant-id": "6500" },
                            body: JSON.stringify(apiPayload),
                            credentials: "include"
                        }).then(r => r.json()).catch(e => ({error: `Antwort ist kein gültiges JSON: ${e.message}`})),
                        args: [API_URL, API_PAYLOAD]
                    }, (results) => {
                        chrome.tabs.remove(tab.id, () => { if (chrome.runtime.lastError) {} });
                        if (chrome.runtime.lastError) {
                            return reject(new Error(chrome.runtime.lastError.message));
                        }
                        const data = results?.[0]?.result;
                        if (data && !data.error) resolve(data);
                        else reject(new Error(data?.error || "Skript-Ausführung fehlgeschlagen"));
                    });
                }
            };
            chrome.tabs.onUpdated.addListener(listener);
        });
    });
}


async function updateBadge() {
    const isLicensed = await validateLicenseInBackground();
    const { tooltipEnabled } = await chrome.storage.local.get({ tooltipEnabled: true });

    if (!isLicensed || !tooltipEnabled) {
        await chrome.action.setBadgeText({ text: '' });
        await chrome.action.setTitle({ title: isLicensed ? "Heide Park Ticket Übersicht (Badge deaktiviert)" : "Heide Park Ticket Übersicht (Lizenz ungültig)" });
        return;
    }

    try {
        const data = await fetchDataViaTab();
        const now = new Date();
        const year = now.getFullYear();
        const month = (now.getMonth() + 1).toString().padStart(2, '0');
        const day = now.getDate().toString().padStart(2, '0');
        const today = `${year}-${month}-${day}`;
        
        const todaysData = data?.SERVICE?.DATES?.D.find(d => d.date === today);
        const platinTicket = todaysData?.PS?.P?.find(p => p.name?.trim().toLowerCase().startsWith("express ticket platin"));

        if (platinTicket) {
            const capacity = parseInt(platinTicket.capacity, 10);
            const available = parseInt(platinTicket.available, 10);
            const sold = capacity - available;
            
            await chrome.action.setBadgeText({ text: sold.toString() });
            await chrome.action.setBadgeBackgroundColor({ color: '#1565c0' });
            await chrome.action.setTitle({ title: `Verkauft: ${sold} Platin` });
        } else {
            await chrome.action.setBadgeText({ text: 'X' });
            await chrome.action.setBadgeBackgroundColor({ color: '#c91926' });
            await chrome.action.setTitle({ title: "Keine Platin-Daten für heute gefunden." });
        }
    } catch (error) {
        await chrome.action.setBadgeText({ text: 'ERR' });
        await chrome.action.setTitle({ title: `Fehler beim Abrufen der Ticketdaten.` });
    }
}

async function createOrUpdateAlarm() {
    const { badgeInterval } = await chrome.storage.local.get({ badgeInterval: 10 });
    const interval = parseInt(badgeInterval, 10);
    if (interval && interval > 0) {
        await chrome.alarms.create('badgeUpdate', { periodInMinutes: interval });
    }
}

chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === 'badgeUpdate') {
        updateBadge();
    }
});

chrome.runtime.onStartup.addListener(() => {
    createOrUpdateAlarm();
    updateBadge();
});
chrome.runtime.onInstalled.addListener(() => {
    createOrUpdateAlarm();
    updateBadge();
});

chrome.storage.onChanged.addListener((changes, namespace) => {
    if (namespace === 'local' && (changes.tooltipEnabled || changes.badgeInterval)) {
        if (changes.badgeInterval) {
            createOrUpdateAlarm();
        }
        updateBadge();
    }
});