document.addEventListener('DOMContentLoaded', () => {
    const container = document.getElementById('data-container');
    const printContainer = document.getElementById('print-container');
    const contentWrapper = document.querySelector('.content-wrapper');
    const loadingMessage = document.getElementById('loading-message');
    const printBtn = document.getElementById('print-btn');
    const exportCsvBtn = document.getElementById('export-csv-btn');
    const productSelect = document.getElementById('product-select');
    const searchInput = document.getElementById('search-input');

    let fullData = null;
    let productMap = {};

    productSelect.addEventListener('change', updateView);
    searchInput.addEventListener('input', updateView);

    printBtn.addEventListener('click', () => {
        const searchTerm = searchInput.value.trim();
        if (searchTerm) {
            prepareAndPrintSearchResults();
        } else {
            prepareAndPrintProductSelection();
        }
    });

    function prepareAndPrintSearchResults() {
        printContainer.innerHTML = '<h1>Suchergebnisse</h1>';
        let hasResults = false;

        document.querySelectorAll('.data-section').forEach(section => {
            if (section.style.display === 'none') return;
            const visibleRows = Array.from(section.querySelectorAll('tbody tr')).filter(row => row.style.display !== 'none');

            if (visibleRows.length > 0) {
                hasResults = true;
                const title = section.querySelector('h2').cloneNode(true);
                const table = section.querySelector('table').cloneNode(true);
                const tbody = table.querySelector('tbody');
                tbody.innerHTML = '';
                visibleRows.forEach(row => {
                    tbody.appendChild(row.cloneNode(true));
                });
                printContainer.appendChild(title);
                printContainer.appendChild(table);
            }
        });

        if (!hasResults) {
            alert("Keine Suchergebnisse zum Drucken gefunden.");
            return;
        }

        contentWrapper.style.display = 'none';
        printContainer.style.display = 'block';
        window.print();
        contentWrapper.style.display = 'block';
        printContainer.style.display = 'none';
    }

    function prepareAndPrintProductSelection() {
        const selectedProductId = productSelect.value;
        if (selectedProductId === 'all') {
            alert("Bitte wählen Sie ein einzelnes Produkt aus der Liste aus oder verwenden Sie die Suche, um zu drucken.");
            return;
        }
        document.querySelectorAll('.data-section').forEach(section => section.classList.remove('printable'));
        const sectionToPrint = productMap[selectedProductId];
        if (sectionToPrint) {
            contentWrapper.classList.add('printable');
            sectionToPrint.classList.add('printable');
            window.print();
            sectionToPrint.classList.remove('printable');
        } else {
            alert("Fehler: Die ausgewählte Sektion konnte nicht gefunden werden.");
        }
    }

    exportCsvBtn.addEventListener('click', () => exportDataAsCsv());

    chrome.storage.local.get('cachedData', (result) => {
        if (result.cachedData) {
            loadingMessage.style.display = 'none';
            fullData = result.cachedData;
            renderData(fullData);
            populateProductSelect();
            updateView();
        } else {
            loadingMessage.textContent = "Keine gespeicherten Daten gefunden.";
        }
    });

    function updateView() {
        const selectedProductId = productSelect.value;
        const searchTerm = searchInput.value.toLowerCase();
        Object.entries(productMap).forEach(([title, section]) => {
            const isProductVisible = selectedProductId === 'all' || selectedProductId === title;
            let sectionHasMatchingRows = false;
            if (!isProductVisible) {
                section.style.display = 'none';
                return;
            }
            const rows = section.querySelectorAll('tbody tr');
            if (rows.length === 0 && searchTerm) {
                sectionHasMatchingRows = false;
            } else if (rows.length === 0 && !searchTerm) {
                sectionHasMatchingRows = true;
            } else {
                rows.forEach(row => {
                    const rowText = row.textContent.toLowerCase();
                    if (rowText.includes(searchTerm)) {
                        row.style.display = '';
                        sectionHasMatchingRows = true;
                    } else {
                        row.style.display = 'none';
                    }
                });
            }
            section.style.display = (isProductVisible && sectionHasMatchingRows) ? 'block' : 'none';
        });
    }

    function populateProductSelect() {
        Object.keys(productMap).sort((a, b) => a.localeCompare(b)).forEach(title => {
            const option = document.createElement('option');
            option.value = title;
            option.textContent = title;
            productSelect.appendChild(option);
        });
    }

    function renderData(data) {
        renderExpressTable(data.express, 'Express Tickets');
        renderTagesAndAbendTable(data.tages, data.halloween, 'Tages- & Abendtickets');
        renderPackageTable(data.peppaParty, 'Peppas Party-Paket 🐷');
        renderPackageTable(data.drachenParty, 'Drachenzähmen Party-Paket 🐉');
        renderPackageTable(data.ghostbustersParty, 'Ghostbusters Party-Paket 👻');
        renderPackageTable(data.heideparkParty, 'Heide Park Party-Paket 🎉');
        renderMazeTable(data.subterraMaze, 'Maze: SubTerra');
        renderMazeTable(data.mortonMaze, 'Maze: Grand Hotel Morton');
        renderMazeTable(data.parasomnisMaze, 'Maze: Parasomnis');
        renderPackageTable(data.fotoPass, 'Foto-Pass 📸');
        renderPackageTable(data.parkplatz, 'Parkplatz 🅿️');
    }

    function createDataSection(title) {
        const section = document.createElement('div');
        section.classList.add('data-section');
        section.id = title.replace(/[^a-zA-Z0-9]/g, '-');
        const titleEl = document.createElement('h2');
        titleEl.textContent = title;
        section.appendChild(titleEl);
        container.appendChild(section);
        productMap[title] = section;
        return section;
    }

    function createTable(section, headers) {
        const table = document.createElement('table');
        const thead = document.createElement('thead');
        const tbody = document.createElement('tbody');
        const headerRow = document.createElement('tr');
        headers.forEach(headerText => {
            const th = document.createElement('th');
            th.textContent = headerText;
            headerRow.appendChild(th);
        });
        thead.appendChild(headerRow);
        table.appendChild(thead);
        table.appendChild(tbody);
        section.appendChild(table);
        return tbody;
    }

    function formatDate(dateStr) {
        if (!dateStr) return '';
        return dateStr.replace(/^(\d{4})-(\d{2})-(\d{2})$/, "$3.$2.$1");
    }

    function formatTime(timeStr) {
        if (!timeStr) return "";
        let [time, modifier] = timeStr.split(' ');
        let [hours, minutes] = time.split(':');
        hours = parseInt(hours, 10);
        if (modifier === 'PM' && hours < 12) hours += 12;
        if (modifier === 'AM' && hours === 12) hours = 0;
        return `${hours.toString().padStart(2, '0')}:${minutes}`;
    }

    function renderExpressTable(data, title) {
        if (!data?.SERVICE?.DATES?.D) return;
        const section = createDataSection(title);
        const tbody = createTable(section, ['Datum', 'Ticket-Typ', 'Verkauft', 'Verfügbar', 'Kapazität', 'Status']);
        const allDays = data.SERVICE.DATES.D.sort((a, b) => a.date.localeCompare(b.date));
        for (const day of allDays) {
            if (day.PS?.P) {
                const tickets = Array.isArray(day.PS.P) ? day.PS.P : [day.PS.P];
                for (const ticket of tickets) {
                    const row = tbody.insertRow();
                    const sold = ticket.soldOut ? ticket.capacity : (parseInt(ticket.capacity) - parseInt(ticket.available));
                    
                    // *** HIER IST DIE ÄNDERUNG ***
                    let ticketName = ticket.name;
                    if (ticketName === "Express Ticket Platin (Aktivierung im Service Center)") {
                        ticketName = "Express Ticket Platin";
                    }

                    row.innerHTML = `<td>${formatDate(day.date)}</td><td>${ticketName}</td><td>${sold}</td><td>${ticket.available}</td><td>${ticket.capacity}</td><td>${ticket.soldOut||ticket.available==="0"?'Ausverkauft':'Verfügbar'}</td>`;
                }
            }
        }
    }

    function renderTagesAndAbendTable(tagesData, halloweenData, title) {
        if (!tagesData?.SERVICE?.DATES?.D && !halloweenData?.SERVICE?.DATES?.D) return;
        const section = createDataSection(title);
        const tbody = createTable(section, ['Datum', 'Ticket-Typ', 'Verkauft', 'Verfügbar', 'Kapazität', 'Status']);
        const combinedData = {};

        const processTages = (day) => {
            if (!day.PS?.P) return;
            if (!combinedData[day.date]) combinedData[day.date] = [];
            const tickets = Array.isArray(day.PS.P) ? day.PS.P : [day.PS.P];
            tickets.forEach(ticket => {
                if (ticket.name === "1-Tages-Ticket") {
                    combinedData[day.date].push(ticket);
                }
            });
        };

        const processHalloween = (day) => {
            if (!day.PS?.P) return;
            if (!combinedData[day.date]) combinedData[day.date] = [];
            const tickets = Array.isArray(day.PS.P) ? day.PS.P : [day.PS.P];
            tickets.forEach(ticket => {
                if (ticket.name && ticket.name.includes("Abendticket Halloween")) {
                    combinedData[day.date].push(ticket);
                }
            });
        };

        (tagesData?.SERVICE?.DATES?.D || []).forEach(processTages);
        (halloweenData?.SERVICE?.DATES?.D || []).forEach(processHalloween);

        const sortedDates = Object.keys(combinedData).sort();
        for (const date of sortedDates) {
            for (const ticket of combinedData[date]) {
                const row = tbody.insertRow();
                const sold = ticket.soldOut ? ticket.capacity : (parseInt(ticket.capacity) - parseInt(ticket.available));
                row.innerHTML = `<td>${formatDate(date)}</td><td>${ticket.name}</td><td>${sold}</td><td>${ticket.available}</td><td>${ticket.capacity}</td><td>${ticket.soldOut||ticket.available==="0"?'Ausverkauft':'Verfügbar'}</td>`;
            }
        }
    }

    function renderPackageTable(data, title) {
        if (!data?.SERVICE?.D || !Array.isArray(data.SERVICE.D)) return;
        const section = createDataSection(title);
        const tbody = createTable(section, ['Datum', 'Verkauft', 'Verfügbar', 'Kapazität', 'Status']);
        const sortedDays = data.SERVICE.D.sort((a, b) => a.date.localeCompare(b.date));
        for (const day of sortedDays) {
            if (day.T) {
                const row = tbody.insertRow();
                row.innerHTML = `<td>${formatDate(day.date)}</td><td>${day.T.used}</td><td>${day.T.available}</td><td>${day.T.capacity}</td><td>${day.T.available==="0"?'Ausverkauft':'Verfügbar'}</td>`;
            }
        }
    }

    function renderMazeTable(data, title) {
        if (!data?.SERVICE?.D || !Array.isArray(data.SERVICE.D)) return;
        const section = createDataSection(title);
        const tbody = createTable(section, ['Datum', 'Uhrzeit', 'Verkauft', 'Verfügbar', 'Kapazität', 'Status']);
        const sortedDays = data.SERVICE.D.sort((a, b) => a.date.localeCompare(b.date));
        for (const day of sortedDays) {
            if (day.T) {
                const slots = Array.isArray(day.T) ? day.T : [day.T];
                const sortedSlots = slots.sort((a, b) => formatTime(a.time).localeCompare(formatTime(b.time)));
                for (const slot of sortedSlots) {
                    const row = tbody.insertRow();
                    row.innerHTML = `<td>${formatDate(day.date)}</td><td>${formatTime(slot.time)}</td><td>${slot.used}</td><td>${slot.available}</td><td>${slot.capacity}</td><td>${slot.available==="0"?'Ausverkauft':'Verfügbar'}</td>`;
                }
            }
        }
    }

    function exportDataAsCsv() {
        if (!fullData) {
            alert("Keine Daten zum Exportieren vorhanden.");
            return;
        }
        let csvContent = "data:text/csv;charset=utf-8,";
        csvContent += "Kategorie;Datum;Name/Uhrzeit;Verkauft;Verfügbar;Kapazität;Status\r\n";
        const addRow = (cat, date, name, sold, avail, cap, status) => {
            csvContent += `"${cat}";"${date}";"${name}";"${sold}";"${avail}";"${cap}";"${status}"\r\n`;
        };
        (fullData.express?.SERVICE?.DATES?.D || []).forEach(d => (Array.isArray(d.PS.P) ? d.PS.P : [d.PS.P]).forEach(t => {
            const sold = t.soldOut ? t.capacity : (parseInt(t.capacity) - parseInt(t.available));
             let ticketName = t.name;
            if (ticketName === "Express Ticket Platin (Aktivierung im Service Center)") {
                ticketName = "Express Ticket Platin";
            }
            addRow('Express', formatDate(d.date), ticketName, sold, t.available, t.capacity, t.soldOut || t.available === "0" ? 'Ausverkauft' : 'Verfügbar');
        }));
        const packageMap = {
            'Maze: SubTerra': fullData.subterraMaze,
            'Maze: Grand Hotel Morton': fullData.mortonMaze,
            'Maze: Parasomnis': fullData.parasomnisMaze,
            'Peppas Party-Paket': fullData.peppaParty,
            'Drachenzähmen Party-Paket': fullData.drachenParty,
            'Ghostbusters Party-Paket': fullData.ghostbustersParty,
            'Heide Park Party-Paket': fullData.heideparkParty,
            'Foto-Pass': fullData.fotoPass,
            'Parkplatz': fullData.parkplatz
        };
        for (const [name, data] of Object.entries(packageMap)) {
            if (data?.SERVICE?.D) {
                (Array.isArray(data.SERVICE.D) ? data.SERVICE.D : [data.SERVICE.D]).forEach(d => {
                    if (d.T) {
                        const slots = Array.isArray(d.T) ? d.T : [d.T];
                        slots.forEach(s => {
                            const subName = s.time ? formatTime(s.time) : name;
                            addRow(name, formatDate(d.date), subName, s.used, s.available, s.capacity, s.available === "0" ? 'Ausverkauft' : 'Verfügbar');
                        });
                    }
                });
            }
        }
        const encodedUri = encodeURI(csvContent);
        const link = document.createElement("a");
        link.setAttribute("href", encodedUri);
        link.setAttribute("download", "heide_park_ticket_daten.csv");
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
});